package dtu.library.ui;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringReader;
import java.io.StringWriter;


public class UtilitiesForTestingUI {
	/**
	 * Tests one screen interaction, i.e. the menu, the input of the user, any possible output as the result
	 * of selecting something. The libraryUI will be called with printMenu, waitForSelection, and processSelection
	 * @param libraryUI
	 * The system under test.
	 * @param expectedMenu
	 * The expected contents (or part thereof) of the printed menu.
	 * @param input
	 * The input the user makes. This is to double check that the input stream provided by the reader is synchronized with the action.
	 * @param expectedOutput
	 * The expected output after the selection has been processed.
	 * @param expectedExitStatus
	 * The expected boolean value returned by the processSelection method. If true that means, the application should exit.
	 * @throws IOException
	 */
	public void testScreenInteraction(LibraryUI libraryUI, String expectedMenu,
			String input, String expectedOutput, boolean expectedExitStatus) throws IOException {
		
		// Get each time a new StringWriter, so that each output can be compared in isolation and not as a string containing the output
		// of past interactions.
		StringWriter out = new StringWriter(); 
		
		// Print the menu on the screen.
		libraryUI.printMenu(new PrintWriter(out));
		
		// Check that the menu contains expectedMenu as substring.
		assertTrue(out.toString().contains(expectedMenu));

		// Read the input from the reader.
		BufferedReader reader = new BufferedReader(new StringReader(input));

		String line = libraryUI.readInput(reader);
		// Check that the expected selection has been made.
		assertEquals(input,line);
		
		// Process the input.
		out = new StringWriter();
		boolean exit = libraryUI.processInput(line, new PrintWriter(out));
		// Check for that output generated by the processing the selection, and
		// check that the exit status is the expected one.
		String eol = System.getProperty( "line.separator" );
		assertEquals(expectedOutput+eol,out.toString());
		assertEquals(expectedExitStatus,exit);
	}
}
