package dtu.library.app;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import java.util.Calendar;
import java.util.GregorianCalendar;

import org.junit.Test;

/**
 * Contains the tests for testing when a book is overdue.<p>
 * This tests also the business logic what happens when a user has overdue books
 * and wants to borrow another book.
 * @author hub
 *
 */
public class TestOverdueCd extends SampleDataSetup {

	public final static int MAX_DAYS_FOR_CD_LOAN = 1 * 7;
	
	/***
	 * Tests the overdue status of a book. A book is overdue, when it is borrowed and
	 * the current date is more than 28 days (4 weeks) later than the borrow date.
	 * <ol>
	 * 	<li>Create a mock object for the date server and tell the library app to use that mock object
	 * 	<li>Make sure that on borrowing the date server returns 15.1.2011
	 * 	<li>Get a user by CPR number and a book by signature and let the user borrow the book
	 *  <li>Compute the date 28 days after the borrow date and set is as the new return value for the mock object
	 *  <li>Check that the book is not yet overdue
	 *  <li>Set the date 29 days after the borrow date
	 *  <li>Check that the book is overdue
	 * </ol>
	 * @throws Exception
	 */
	@Test
	public void testOverdueCd() throws Exception {

		// Step 1
		DateServer dateServer = mock(DateServer.class);
		libApp.setDateServer(dateServer);

		// Step 2
		Calendar cal = new GregorianCalendar(2011,Calendar.JANUARY,10);
		when(dateServer.getDate()).thenReturn(cal);

		// Step 3
		String cprNumber = "1234651234";
		User user = libApp.userByCprNumber(cprNumber);

		String signature = "CD1";
		Medium cd = libApp.mediaBySignature(signature);

		user.borrowMedium(cd);

		// Step 4
		Calendar newCal = new GregorianCalendar();
		newCal.setTime(cal.getTime());
		newCal.add(Calendar.DAY_OF_YEAR, MAX_DAYS_FOR_CD_LOAN);
		when(dateServer.getDate()).thenReturn(newCal);

		// Step 5
		assertFalse(cd.isOverdue());

		// Step 6
		newCal = new GregorianCalendar();
		newCal.setTime(cal.getTime());
		newCal.add(Calendar.DAY_OF_YEAR, MAX_DAYS_FOR_CD_LOAN + 1);
		when(dateServer.getDate()).thenReturn(newCal);

		// Step 7
		assertTrue(cd.isOverdue());
	}

	/***
	 * Test that the user can't borrow a book if he has an overdue book.
	 * <ol>
	 * 	<li>The user borrows a book
	 * 	<li>The date is set to 29 days after the book has been borrowed
	 * 	<li>The user tries to borrow another book
	 * 	<li>The system throws a HasOverdueBooksException
	 * </ol>
	 * @throws Exception
	 */
	@Test
	public void testBorrowWithOverdueBooks() throws Exception {
		
			// Step 1
			DateServer dateServer = mock(DateServer.class);
			libApp.setDateServer(dateServer);

			Calendar cal = new GregorianCalendar(2011, Calendar.JANUARY, 1);

			when(dateServer.getDate()).thenReturn(cal);

			String cprNumber = "1234651234";
			User user = libApp.userByCprNumber(cprNumber);

			String signature = "CD1";
			Medium cd = libApp.mediaBySignature(signature);

			user.borrowMedium(cd);

			// Step 2
			Calendar newCal = new GregorianCalendar();
			newCal.setTime(cal.getTime());
			newCal.add(Calendar.DAY_OF_YEAR,  MAX_DAYS_FOR_CD_LOAN + 1);
			when(dateServer.getDate()).thenReturn(newCal);

			assertTrue(cd.isOverdue());
			
			// Step 3
			signature = "book1";
			cd = libApp.mediaBySignature(signature);
			
			try {
				user.borrowMedium(cd);
				fail("Should throw HasOverdueBookException");
			} catch (HasOverdueMedia e) {
				// Step 4
				assertEquals("User has overdue media",e.getMessage());
				assertFalse(user.getBorrowedMedia().contains(cd));
			}
	}
}
